(in-package :lcc)

(declaim (optimize (debug 3) (speed 0)))

(eval-when (:load-toplevel :compile-toplevel)
  (use-package :cl-gsl-rng))

(export '(make-zipf-distribution sample-from
	  <zipf-distribution>))

;;; {{{ Random number generation: GSL interface
(defparameter *rng-struct* (make-generator "gsl_rng_mt19937"))

;; XXX Perhaps for some reason, but probably for none, make-generator
;; returns a struct with one member.  The rest of the functions don't
;; want the struct, they want the member.  So let's just take it out
;; now.  Must submit-patch/shoot-author.
(defparameter *rng*  (cl-gsl-rng::rng-ptr *rng-struct*))
;;; }}}
;;; {{{ Protocol
(defgeneric random-number (distribution)
  (:documentation "Return a random number from the cumulative
probability distribution CPD."))

(defgeneric sample-from (distribution number-of-samples))
;;; }}}
;;; {{{ probability distribution
(defclass <distribution> ()
  ())
;;; }}}
;;; {{{ discrete probability distribution
(defclass <discrete-distribution> ()
  ((pdf :accessor pdf-of)
   (cpd :accessor cpd-of)))

(defmethod initialize-instance :after ((d <discrete-distribution>)
				       &key &allow-other-keys)
  ;; I think the quickest way to do this is generate a vector of the CPD
  ;; (normalized, of course!), and scan it until we find the right
  ;; value.  Or do a binary search.
  (let* ((pdf (pdf-of d))
	 (cpd (make-array (list (length pdf)) :element-type 'double-float)))
    (setf (cpd-of d) cpd)
    (setf (aref cpd 0) (aref pdf 0))
    (loop for i from 1 to (- (length pdf) 1)
       do (setf (aref cpd i) (+ (aref cpd (- i 1))  (aref pdf i))))))

(defmethod sample-from ((d <discrete-distribution>) n)
  (let ((samples (make-array (list n))))
    (loop for i from 0 to (decf n)
       do (setf (aref samples i) (random-number d)))
    samples))

(defmethod random-number ((d <discrete-distribution>))
  (let ((r (gsl-rng::gsl-rng-uniform *rng*)))
    (position-if (lambda (x) (< r x)) (cpd-of d))))
;;; }}}
;;; {{{ Zipf distribution

(defclass <zipf-distribution> (<discrete-distribution>)
  ((s :initarg :s)
   (n :initarg :n)))

(defmethod shared-initialize ((d <zipf-distribution>) slot-names
			      &rest initargs &key &allow-other-keys)
  (call-next-method)
  (setf (pdf-of d)
	(with-slots (n s) d
	 (let ((v (make-array (list n) :element-type 'double-float)))
	   (loop for i from 1 to n
	      do (setf (aref v (- i 1))
		       (coerce (zipf-p i s n) 'double-float)))
	   (normalize v)))))

(defun zipf-p (k s n)
  (/ (/ 1 (expt k s))
     (harmonic n s)))

;;; }}}
;;; {{{ Testing
(defun graph (cpd samples)
  "For graphing a distribution"
  (let* ((sample (sample-from cpd samples))
	 (vec (make-array (length cpd) :initial-element 0))
	 (scaling (/ 100.0 samples)))
    (loop for s across sample
       do (incf (aref vec (coerce s 'integer))))
    (loop for x from 0 to (- (length vec) 1)
       do (progn
	    (format t "~%~3d |" x)
	     (loop for y from 1 to (ceiling (* scaling (aref vec x)))
		do (format t "-"))))
    'vec))

;;; }}}
;;; {{{ Sundry

(defun normalize (vec)
  (let* ((n (length vec))
	(sum (loop for i from 0 to (- n 1)
		sum (aref vec i))))
    (loop for i from 0 to (- n 1)
       do  (setf (aref vec i) (/ (aref vec i) sum)))
    vec))

(defun harmonic (n s)
  (loop for i from 1 to n
     sum (/ 1 (expt n s))))

;;; }}}
