(in-package :lcc)

;;; {{{ Matchmaker policy

(defrule set-matchmaking-policy/3
  (<- (set-matchmaking-policy ?protocol ?policy ?protocol+)
      (matchmaking-policies ?policies)
      (member ?policy ?policies)
      (protocol-add common-knowledge ?protocol
		    (known ? (matchmaking-policy ?policy)) ?protocol+)))

(defrule get-matchmaking-policy/2
  (<- (get-matchmaking-policy ?protocol ?policy)
      (protocol-member common-knowledge ?protocol (known ? (matchmaking-policy ?policy)))))

;; We support two policies at the momement.
(defrule matchmaking-policies/1
  (<- (matchmaking-policies (random-incremental ic-incremental ic-joint))))

;;; }}}
;;; {{{ Agent selection

(defrule add-collaborator/4
  (<- (add-collaborator ?role ?id ?p ?p)
      (get-collaborator ?role ?id ?p)
      !)
  (<- (add-collaborator ?role ?id ?p ?pnew)
      (not (get-collaborator ?role ?id ?p))
      (protocol-add common-knowledge ?p (known ? (collaborator ?role ?id)) ?pnew)))

(defrule add-collaborators/3
  (<- (add-collaborators () ?p ?p))
  (<- (add-collaborators ((collaborator ?role ?id) . ?rest) ?p ?pnew)
      (add-collaborator ?role ?id ?p ?pmid)
      (add-collaborators ?rest ?pmid ?pnew)))

(defrule get-collaborator/3
  (<- (get-collaborator ?role ?id ?protocol)
      (protocol-member common-knowledge ?protocol (known ? (collaborator ?role ?id)))))

(defrule get-collaborators/2
  (<- (get-collaborators ?protocol ?collaborators)
      (setof (collaborator ?role ?id) (get-collaborator ?role ?id ?protocol) ?collaborators)
      !)
  (<- (get-collaborators ?protocol ())))

(defrule ensure-collaborator/3
  (<- (ensure-collaborator ?agent ?protocol ?protocol+)
      (get-matchmaking-policy ?protocol ?policy)
      (recruit ?policy ?agent ?protocol ?protocol+))
  (<- (ensure-collaborator ?agent ?protocol ?protocol+)
      (not (get-matchmaking-policy ?protocol ?policy))
      (error "Could not find a matchmaking policy." ())))

;; So, we have one call, recruit/4.  The policy should be "all" or "one".  
;; recruit(Policy, a(Role, Agent), +Pin, ?Pout).
(defrule recruit/4
  (<- (recruit ? (a ? ?id) ?p ?p) 
      (nonvar ?id))
  (<- (recruit ? (a ?role ?id) ?p ?p) 
      (var ?id)
      (get-collaborator ?role ?id ?p) 
      ;;(warning recruit/4 "reusing agent ~S for role of ~S~%" (?id ?role))
      )
  (<- (recruit random-incremental (a ?role ?id) ?p ?pnew)
      (var ?id) 
      (not (get-collaborator ?role ?id ?p))
      (get-protocol-name ?p ?protocol-name)
      (agents-for-role ?protocol-name ?role ?ids)
      (choose-one-randomly ?ids ?id)
      (add-collaborator ?role ?id ?p ?pnew))
  (<- (recruit ic-joint (a ?role ?id) ?p ?pnew)
      (matchmake-joint ?p ?pnew))
  (<- (recruit ic-incremental (a ?role ?id) ?p ?pnew) 
      (not (get-collaborator ?role ?id ?p))
      (get-collaborators ?p ?collaborators)
      (get-protocol-name ?p ?protocol-name)
      (agents-for-role ?protocol-name ?role ?avail) 
      (setof (ic-score ?id ?quality)
	     (and (member ?id ?avail)
		  (quality (collaborator ?role ?id) ((protocol ?protocol-name) . ?collaborators)  ?quality))
	     ?unsorted)
      ;;    (format "recruit-ic-inc: scores=~S~%" (?unsorted))
      (rank ?unsorted ?sorted)
      (pickone ?sorted (ic-score ?id ?))
      (add-collaborator ?role ?id ?p ?pnew))
  (<- (recruit numeric-probability-incremental (a ?role ?id) ?p ?pnew) 
      (not (get-collaborator ?role ?id ?p))
      (get-collaborators ?p ?collaborators)
      (get-protocol-name ?p ?protocol-name)
      (agents-for-role ?protocol-name ?role ?avail) 
      (setof ?id (collaborator ?role ?id) ?candidates)
      (ranked-numeric-probability ?candidates ?protocol-name ?collaborators ?sorted)
      (pickone ?sorted (ic-score ?id ?))
      (add-collaborator ?role ?id ?p ?pnew)))

;; (define-pill-function ranked-numeric-probability/3
;;     (?+candidates ?+protocol-name ??answer cont)
;;   (let ((protocol-name (paiprolog::deref-exp ?+protocol-name))
;; 	(candidates (paiprolog::deref-exp ?+candidates)))
;;     (paiprolog::unify! ??answer (lcc::numeric-probability protocol-name candidates)))
;;     (funcall cont)))

;; (defun numeric-probability (protocol-name candidates)
;;   (labels ((candidate-probs (candidate)
;; 	     (ic:ic-probability `(:cond (paiprolog::outcome paiprolog::good) (:and ,candidate (paiprolog::protocol ,protocol-name))))))
;;     (let* ((unranked (mapcar #'candidate-probs candidates)))
;;       (mapcar #'(lambda (name score) `(paiprolog::ic-score ,(third name)
;; 						      (paiprolog::/ ,(car score) ,(cdr score))))
;; 	      candidates unranked))))

;;; }}}
;;; {{{ Role definition 

;;XXX Has to be fixed to enable more role definitions.
(defmacro define-role (name clause)
  (let ((clause (paiprolog::make-anonymous clause)))
    `(<- (%role-definition ,name ,clause))))

;; The ?role-type is whether it's (travel-agent ?from ?to) or some
;; such, the pattern that's going to be matched at message send time.
;; The ?label is the name of a particular role definition.  XXX We
;; should include arity information at some point, too!
(defrule role-definition/3
  (<- (role-definition ?role ?label ?definition)
      (var ?role)
      (error "role-definition: ?role is a variable."))
  (<- (role-definition ?role ?label ?definition)
      (nonvar (?role))
      (format "role-definition: ~W~%" ?role)
      (%role-definition ?label (== (a ?role ?id) ?body))
      (independent-freshen (== (a ?role ?id) ?body) ?definition)))

;; All possible role definitions for ?role.
(defrule role-candidates/2
  (<- (role-candidates ?role ?canditates)
      (setof (role-definition ?label ?def)
	     (role-definition ?role ?label ?def)
	     ?canditates)))

(defrule role-defined?/2
  (<- (role-defined? ?role (def ? ?framework ?))
      (member (== (a ?role ?) ?) ?framework)
      !))

(defrule ensure-role-definition/3
  (<- (ensure-role-definition (a ?role ?id) ?protocol ?protocol)
      (role-defined? ?role ?protocol))
  (<- (ensure-role-definition (a ?role ?id) ?protocol ?protocol+)
      (not (role-defined? ?role ?protocol))
      (get-matchmaking-policy ?protocol ?policy)
      (ensure-role-definition ?policy ?role ?protocol ?protocol+))
  (<- (ensure-role-definition ?role ?protocol ?)
      (not (get-matchmaking-policy ?protocol ?policy))
      (error "Could not find a matchmaking policy." ())))

(defrule ensure-role-definition/4
  (<- (ensure-role-definition ic-incremental ?role ?protocol ?protocol+)
      (select-role ?role ?protocol ?role-definition)
      (add-role-definition ?role-definition ?protocol ?protocol+))
  (<- (ensure-role-definition random-incremental ?role ?protocol ?protocol+)
      (role-candidates ?role ?candidates)
      (choose-one-randomly ?candidates ?role-definition)
      (add-role-definition ?role-definition ?protocol ?protocol+))
  (<- (ensure-role-definition ?policy ?role ?protocol ?protocol+)
      (not (member ?policy (random-incremental ic-incremental)))
      (error "Not a legitimate policy." ())))

(defrule select-role/3
  (<- (select-role ?role ?p (role-definition ?label ?role-definition)) 
      (get-role-definitions ?p ?definitions)
      (get-protocol-name ?p ?protocol-name)
      (setof (ic-score ?definition-name ?quality)
	     (and (role-definition ?role ?definition-name ?)
		  (quality (role-definition ?role ?definition-name)
			   ((protocol ?protocol-name) . ?definitions) ?quality))
	     ?unsorted)
      (rank ?unsorted ?sorted)
      (pickone ?sorted (ic-score ?label ?))
      (role-definition ?role ?label ?role-definition)))

(defrule add-role-definition/3
  (<- (add-role-definition (role-definition ?label (== (a ?role ?id) ?definition)) ?protocol ?protocol++)
      (= ?protocol (def ?clauses ?framework ?knowledge))
      (= ?protocol+ (def ?clauses ((== (a ?role ?id) ?definition) . ?framework) ?knowledge))
      (protocol-add common-knowledge ?protocol+ (known ? (role-definition ?role ?label)) ?protocol++)))

(defrule get-role-definition/3
  (<- (get-role-definition ?role ?label ?protocol)
      (protocol-member common-knowledge ?protocol (known ? (role-definition ?role ?label)))))

(defrule get-role-definitions/2
  (<- (get-role-definitions ?protocol ?role-definitions)
      (setof (role-definition ?role ?id) (get-role-definition ?role ?id ?protocol) ?role-definitions)
      !)
  (<- (get-role-definitions ?protocol ())))

;;; }}}
;;; {{{ Selecting using the IC
;;; {{{ Prolog interface to incidence calculus 

(defparameter *database* nil
  "The incidence calculus database.")

(define-pill-function reset-matchmaker-database/0 (cont)
  "Destroys all knowledge of worlds."
  (setf *database* (ic:make-database))
  (funcall cont))

(define-pill-function tell/2 (?+pred ?+world cont)
  "Add WORLD to the worlds in which PRED is true."
  (ic:tell *database* (paiprolog::deref-exp ?+pred) (paiprolog::deref-exp ?+world))
  (funcall cont))

(define-pill-function ask/2 (?+sentence ??worlds cont)
  "Return list of worlds in which SENTENCE is true"
  (paiprolog::unify! (ic:ask *database* (paiprolog::deref-exp ?+sentence)) ??worlds)
  (funcall cont))

(define-pill-function probability/2 (?+sentence ??probability cont)
  (let* ((sent (paiprolog::deref-exp ?+sentence))
	 (prob (ic:probability *database* sent)))
      (paiprolog::unify! ??probability
			 (list 'paiprolog::/ (car prob) (cdr prob))))
  (funcall cont))

;;; }}}
;;; {{{ Ranking and selection
(defrule pickone/2
  (<- (pickone (?top-ranked . ?rest) ?thing) 
      (random 0 10 ?x)
      (if (= ?x 9)
	  ;; Used to select randomly from only ?rest, but this causes
	  ;; random to blow up if ?rest is empty.
	  (choose-one-randomly (?top-ranked . ?rest) ?thing)
	  (= ?thing ?top-ranked))))

(defrule rank/2
  (<- (rank ?unsorted ?sorted) 
      ;; Because the default ordering seems to cause unfavourable
      ;; advantage to ic-joint in particular, we'll jumble it up first.
      (jumble ?unsorted ?jumbled)
      (merge-sort incidence-rating-betterthan ?jumbled ?sorted)
      !))

(defrule incidence-rating-betterthan/2
  (<- (incidence-rating-betterthan (ic-score ? (/ ?n1 ?d1)) (ic-score ? (/ ?n2 ?d2)))
      (= ?threshold 8)
      (if (= ?d1 0)
	  (is ?q1 0)
	  (is ?q1 (/ ?n1 ?d1)))
      (if (= ?d2 0)
	  (is ?q2 0)
	  (is ?q2 (/ ?n2 ?d2)))
      (or (< ?d1 ?threshold)
	  (> ?q1 ?q2))))

(defrule quality/3
  (<- (quality ?thing ?priors ?quality)
      ;;(format "quality: ~S ~S~%" (?thing ?priors))
      (probability (:cond (outcome good) (:and ?thing . ?priors)) ?quality)
      !)
  (<- (quality ? ? (/ 0 0))))
;;; }}}
;;; }}}
;;; {{{ Incident recording

(defrule record-matchmaking-outcome/2
  (<- (record-matchmaking-outcome ?protocol ?outcome)
      (gensym incident ?world)
      (tell t ?world) 
      (tell (outcome ?outcome) ?world)
      (get-protocol-name ?protocol ?name)
      (tell (protocol ?name) ?world) 
      (get-collaborators ?protocol ?collaborators)
      (record-collaborators ?collaborators ?world)
      (get-role-definitions ?protocol ?role-definitions)
      (record-role-definitions ?role-definitions ?world)))

(defrule record-collaborators/2
  (<- (record-collaborators () ?world))
  (<- (record-collaborators ((collaborator ?role ?id) . ?collaborators) ?world) 
      (msg record-collaborators/2 "2.1~%" ())
      (tell (collaborator ?role ?id) ?world)
      (msg record-collaborators/2 "2.2~%" ())
      (record-collaborators ?collaborators ?world)))

(defrule record-role-definitions/2
  (<- (record-role-definitions () ?world))
  (<- (record-role-definitions ((role-definition ?role ?label) . ?defs) ?world) 
      (tell (role-definition ?role ?label) ?world)
      (record-role-definitions ?defs ?world)))

;;; }}}
;;; {{{ Matchmake-Joint

(defun permute-agent/roles (list)
  "Taking a list ((r1 a1 a2) (r2 a3 a4 a5) (r3 a6)) return ... "
  (if (null list)
      '(())
      (let* ((roles/agents (first list))
	     (permutations (permute-agent/roles (rest list)))
	     (role (first roles/agents)))
	(apply #'append
	       (mapcar #'(lambda (agent)
			   (mapcar (lambda (perm)
				     (cons `(paiprolog::collaborator ,role ,agent) perm))
				   permutations))
		       (rest roles/agents))))))

(define-pill-function permute-agent-roles/2 (?+agent/roles ?permutations cont)
  (let ((a/r (paiprolog::deref-exp ?+agent/roles)))
    (paiprolog::unify! (permute-agent/roles a/r) ?permutations))
  (funcall cont))

(defrule matchmake-joint/2
  (<- (matchmake-joint ?protocol ?protocol-new)
      (get-protocol-name ?protocol ?protocol-name)
      (matchmake-roles ?protocol-name ?roles)
      (generate-options ?protocol-name ?roles ?agent/roles)
      (permute-agent-roles ?agent/roles ?permutations)
      (compute-scores ?protocol-name ?permutations ?scored)
      (rank ?scored ?sorted)
      (pickone ?sorted (ic-score ?collaboration ?))
      (lineout ?collaboration ?sorted)
      (add-collaborators ?collaboration ?protocol ?protocol-new)))

(defrule generate-options/3
  (<- (generate-options ?protocol-name () ()))
  (<- (generate-options ?protocol-name (?role . ?roles) ((?role . ?agents) . ?others))
      (agents-for-role ?protocol-name ?role ?agents)
      (generate-options ?protocol-name ?roles ?others)))

(defrule compute-scores/3
  (<- (compute-scores ?protocol-name ?permutations ?scores)
      (setof (ic-score ?collab ?prob)
	     (and (member ?collab ?permutations)
		  (probability (:cond (outcome good)
				      (:and (protocol ?protocol-name)
					    . ?collab))  ?prob))
	     ?scores)))

;;; }}}

