(in-package :incidence-calculus)

(declaim (optimize (speed 3) (safety 3) (debug 0)))

(defclass <database> ()
  ((table :accessor table-of :initform (make-hash-table :test #'equal))
   (freshest-world :accessor freshest-world-of :initform 0)))

;;; {{{ Lisp interface ask/tell interface

(defun make-database ()
  (make-instance '<database>))

(defun tell (database predicate world)
  "Add WORLD to the worlds in which PRED is true."
  (declare (integer world))
  (let ((worlds (table-of database)))
    (setf (gethash predicate worlds)
	  (merge 'list (gethash predicate worlds) (list world) #'<))))

(defun ask (database exp)
  "List of worlds that satisfy EXP."
  (the list (incidence database exp)))

(defun incidence (database sentence)
  "Return the worlds satsifying SENTENCE."
  (let ((worlds (table-of database)))
    (if (atom sentence)
	(the list (gethash sentence worlds))
	(case (first sentence)
	  ((:and) (funcall #'ic-intersection
			   (mapcar #'(lambda (thing) (incidence database thing))
				   (rest sentence))))
	  ((:or) (funcall #'ic-union
			  (mapcar #'(lambda (thing) (incidence database thing))
				  (rest sentence))))
	  (otherwise (the list (gethash sentence worlds)))))))

(defun safe-/ (a b)
  (if (= 0 b)
      nil
      (/ a b)))

(defun probability (database sentence)
  (if (and (listp sentence) (eq (first sentence) :cond))
      ;; Computing a conditional probability
      (let ((a (ask database (second sentence)))
	    (b (ask database (third sentence))))
	(cons (length (ic-intersection (list a b))) (length b)))
      (cons (length (ask database sentence)) (length (ask database t)))))

(defun fresh-world (database)
  "Return a new, never used world identifier. (of course, you
have to use this all the time, otherwise they won't be so
unique!)"
  (incf (freshest-world-of database)))

;;; }}}
;;; {{{ Pairwise/log/heap operations

(defun blend-with (fn sets)
  "Apply FN to two members of SETS at a time, in a tree fashion."
  (declare (list sets))
  (if (= 1 (length sets))
      (the list (first sets))
      (the list
	(blend-with fn (append (nthcdr 2 sets)
			       (list (funcall (the function fn)
					      (the list (first sets))
					      (the list (second sets)))))))))

(defun log-intersection (sets)
  (the list (blend-with #'ordered-intersection sets)))

(defun log-union (sets)
  (the list (blend-with #'ordered-union sets)))

(defun ordered-intersection (as bs)
  (labels ((f (as bs acc)
	     (if (or (null as) (null bs))
		 (nreverse acc)
		 (let ((a (first as))
		       (b (first bs)))
		   (declare (fixnum a b))
		   (cond ((< a b) (f (rest as) bs acc))
			 ((> a b) (f as (rest bs) acc))
			 (t (f (rest as) (rest bs) (cons a acc))))))))
    (f as bs '())))

(defun ordered-union (as bs)
  (labels ((f (as bs acc)
	     (cond ((null as) (nconc (nreverse acc) bs))
		   ((null bs) (nconc (nreverse acc) as))
		   (t (let ((a (first as))
			    (b (first bs)))
			(declare (fixnum a b))
			(cond ((< a b) (f (rest as) bs (cons a acc)))
			      ((> a b) (f as (rest bs) (cons b acc)))
			      (t (f (rest as) (rest bs) (cons a acc)))))))))
    (f as bs '())))

;;; }}}
;;; {{{ A slow way of computing sets.

(defun insert-set (set sets)
  ;; Place SET into SETS such that (first sets) is under
  ;; #'<.
  (if (null set)
      sets
      (merge 'list (list set) sets #'(lambda (a b)
				  (< (the fixnum (first a))
				     (the fixnum (first b)))))))
(defun sort-sets (sets)
  "Return SETS such that they are sorted by their first element."
  (let ((sorted '()))
    (dolist (s sets) (setf sorted (insert-set s sorted)))
    sorted))

;; At zero cross over, this operation takes time O(number-of-sets).
;; As crossover increases, it becomes insertion sort, so has time
;; O(n^2).  We could go to log sorts at this point.
(defun pick-off (el sets)
  (if (null sets)
      '()
      (insert-set (if (= (the fixnum el) (the fixnum (first (first sets))))
		      (rest (first sets))
		      (first sets))
		  (pick-off el (rest sets)))))

(defun multi-union (sets union)
  (if (null sets)
      (nreverse union)
      (let ((thing (first (first sets))))
	(multi-union (pick-off thing sets) (cons thing union)))))


(defun multi-intersection (sets intersection)
  (declare (list sets))
  (labels ((stop? (sets)
	     (or (null sets)
		 (position-if #'null sets))))
    (if (stop? sets)
	(nreverse intersection)
	(let ((el (the fixnum (first (first sets)))))
	  (if (every (lambda (set) (= el (the fixnum (first set))))
		     (the list (rest sets)))
	      (multi-intersection (mapcar #'rest sets) (cons el intersection))
	      (let ((new (mapcar (lambda (set) (if (= (the fixnum (first set)) el)
					      (rest set)
					      set))
				 sets)))
		(if (stop? new)
		    (nreverse intersection)
		    (multi-intersection (sort-sets new) intersection))))))))

;;; }}}
;;; {{{ Bind internal functions to exported names

(setf (symbol-function 'ic-intersection) #'log-intersection)
(setf (symbol-function 'ic-union) #'log-union)

;;; }}}
