;; A company (me) wants to IPO.  We get a bank and an accountants to
;; price us.  If they agree, and don't look ridiculous, we IPO.  If
;; not, we which inform the market that the two companies are dodgy.
;; We guess, probabilistically, whether the price is too high.  My
;; intent here is to frame a couple of good outcomes...  Illegal agents
;; are more likely to get a good price, but more likely to get caught.
;; A mix of legal and illegal agents will never succeed at IPO, because
;; they'll argue before the stock market ever sees the price.  We use
;; an oracle for convinience.

;; We are a company, looking to sell ourselves.  Our criteria are good
;; price and legality!

;; It's interesting that we (or at least, I) find it difficult to think
;; of properly distributed scenarios.  Everything seems to find a
;; central player who does more than their fair share.
(in-package :lcc)

;; Each agent is ranked by their compentance and legality.  (Neither of
;; which, of course!, they are going to declare in their services'
;; semantics).

;; agent(Name, Roles, Quality, Legality) :-
;; 	member(Competence, [novice, competent, expert]).
;; 	member(Legal, [criminal, fair, squeekyclean]).

;; :- multifile framework/3, hiveagents/2, agent_roles/2.
;; :- use_module(library(random)).

;;(framework ipo company (accountant bank)))

(define-framework ipo
    ((a company ?c) ==
     (ask (company-value ?c)) => (a accountant ?a) then
     (tell (company-value ?c ?av)) <= (a accountant ?a) then
     (ask (company-value ?c)) => (a bank ?b) then
     (tell (company-value ?c ?bv)) <= (a bank ?b) then
     (((tell (ipo-value ?c ?ipov)) => (a market ftse) <-- (ipo-value-ok ?av ?bv ?ipov) then
       (tell (brokering good ipo ?collaborators)) => (a broker broker) <-- (setof (collaborator ?r ?ag) (collaborator ?r ?ag) ?collaborators))
      or
      ((tell (naughty ?a ?b)) => (a market ftse) then
       (tell (brokering bad ipo ?collaborators)) => (a broker broker)
       <-- (setof (collaborator ?r ?ag) (collaborator ?r ?ag) ?collaborators))))

  ((a accountant ?a) ==
   (ask (company-value ?c)) <= (a company ?c) then
   (tell (company-value ?c ?v)) => (a company ?c) <-- (valuation ?a ?c ?v))
  
  ((a bank ?b) ==
   (ask (company-value ?c)) <= (a company ?c) then
   (tell (company-value ?c ?v)) => (a company ?c) <-- (valuation ?b ?c ?v))
  
  ((a market ?m) ==
   ((tell (ipo-value ?c ?ipov)) <= (a company ?c)
    or
    (tell (naughty ?a ?b)) <= (a company ?c)))

  ((a broker broker) ==
   (record ?protocol ?outcome ?collaborators) <--
   (tell (brokering ?outcome ?protocol ?collaborators)) <= (a company ?)))


(<- (hiveagents ipo ?all)
    (setof ?company (oracle-price ?company ?price) ?companies)
    (setof ?agent (advisor ?agent ?roles ?criminality) ?agents)
    (append ?agents ?companies ?most)
    (append ?most (ftse broker) ?all))

(<- (agent-roles ?agent ?roles)
  (advisor ?agent ?roles ?))

(<- (advisor andersons (accountant) criminal))
(<- (advisor deloitte (accountant) fair))
(<- (advisor pwc (accountant) squeekyclean))

(<- (advisor banco-mafioso (bank) criminal))
(<- (advisor budget-bank (bank) squeekyclean))
(<- (advisor bank-of-jockland (bank) fair))

(<- (agents-for-role ipo ?role ?agents)
     (setof ?agent (and (advisor ?agent ?roles ?niceness)
			(member ?role ?roles))
	    ?agents))

;; This set of spreads is wider.  I think, supposedly, this would am
(<- (spread criminal 1.3 2.0))
(<- (spread fair 1.0 1.3))
(<- (spread squeekyclean 0.9 1.1))

;; (<- (spread criminal 1.2 1.4))
;; (<- (spread fair 1.0 1.2))
;; (<- (spread squeekyclean 0.8 1.0))

;;; ?price in billions
(<-- (oracle-price cocacola 67))
(<- (oracle-price microsoft 61))
(<- (oracle-price ibm 54))
(<- (oracle-price ge 44))
(<- (oracle-price intel 34))
(<- (oracle-price disney 27))
(<- (oracle-price mcdonalds 25))
(<- (oracle-price nokia 24))
(<- (oracle-price toyota 23))
(<- (oracle-price marlboro 22))

(<- (valuation ?valuer ?company ?value)
    (oracle-price ?company ?true-value)
    (advisor ?valuer ? ?legality)
    (spread ?legality ?lower ?upper)
    (random ?lower ?upper ?factor)
    (is ?value (* ?factor ?true-value)))

(<- (ipo-value-dodgy ?v1 ?v2)
    (not (ipo-value ?v1 ?v2 ?)))

(<-- (ipo-value-ok ?v1 ?v2 ?final-v)
     (is ?final-v (/ (+ ?v1 ?v2) 2))
     (is ?d1 (abs (- ?final-v ?v1)))
     (is ?d2 (abs (- ?final-v ?v2)))
     (is ?margin (/ ?final-v 8))
     (< ?d1 ?margin)
     (< ?d2 ?margin))

;;; You could have a choice of markets wherein some banks would be
;;; better at taking companies to the NASDAQ, others to the Dow Jones.

(<-- (framework-agents ipo ?agents)
     (setof ?c (oracle-price ?c ?) ?companies)
     (choose-one-randomly ?companies ?company)
     ! 
     (agents-for-role ipo accountant ?as) 
     (setof (a accountant ?x) (member ?x ?as) ?ass)
     (agents-for-role ipo bank ?bs) 
     (setof (a bank ?x) (member ?x ?bs) ?bss) 
     (append ?ass ?bss ?valuers) 
     (append ((a company ?company) (a market ftse) (a broker broker)) ?valuers ?agents))

;; framework_agents(ipo, Agents) :-
;;     setof(C, oracle_price(C,_), Companies),
;;     choose_one_randomly(Companies, Company), !,
;;     agents_for_role(accountant,As),
;;     setof(a(accountant, X), member(X, As), Ass),
;;     agents_for_role(bank,Bs),
;;     setof(a(bank, X), member(X, Bs), Bss),
;;     append(Ass, Bss, Valuers),
;;     append([a(company, Company), a(market, ftse), a(matchmaker,matchmaker)],
;; 	   Valuers, Agents).
