;; We believe that hotels are more constrained than flights, so that
;; it will be more easy to find a hotel, then a flight than vice
;; versa.  Another way might be to constrain the cost of the travel.

(in-package :lcc)

;;; {{{ Protocol

;; ?tolerance specifies the number of days ahead or behind their
;; intended dates the customer will accept.  Once a particular 

(define-framework holiday
    (== (a traveller ?t)
	(then (<-- (=> (book-holiday ?src ?dst ?start ?end ?money) (a travel-agent ?ta))
		   (travel-details ?src ?dst ?start ?end ?money))
	      (or (then (<= (booking ?start ?end ?cost) (a travel-agent ?ta))
			(=> (matchmaking good) (a matchmaker matchmaker)))
		  (then (<= failure (a travel-agent ?ta))
			(=> (matchmaking bad) (a matchmaker matchmaker))))))

  ;; No travel-agent role has been specified here!  We leave it to the
  ;; matchmaker to find one.  In reality, there would be no hotel or
  ;; airline roles either, since they are used only by the
  ;; (unspecified!) travel-agent.

  ;; XXX We might want to specify, for instance, that we use a railway
  ;; rather than an airline.
  
  (== (a hotel ?hotel)
      (then (<= (book-hotel ?location ?start ?end ?money) (a ?r ?a))
	    (or (<-- (=> (hotel-booking ?cost) (a ?r ?a))
		     (room-available ?location ?start ?end ?money ?cost))
		(=> no-vacancy (a ?r ?a)))))

  (== (a airline ?airline)
      (then (<= (book-flight ?src ?dst ?start ?end ?money) (a ?r ?a))
	    (or (<-- (=> (flight-booking ?cost) (a ?r ?a))
		     (flight-available ?src ?dst ?start ?end ?money ?cost))
		(=> no-flights (a ?r ?a))))))

;; Clear out all previous %role-definition's
(eval-when (:compile-toplevel :load-toplevel)
  (paiprolog::retract-predicate 'paiprolog::%role-definition 2))

(define-role flight-then-hotel
    (== (a travel-agent ?ta)
	(then (<= (book-holiday ?src ?dst ?start ?end ?money) (a ?booker ?b))
	      (then (=> (book-flight ?src ?dst ?start ?end ?money) (a airline ?airline))
		    (or (then (<= no-flights (a airline ?airline))
			      (=> failure (a ?booker ?b)))
			(then (<= (flight-booking ?flight-cost) (a airline ?airline))
			      (then (<-- (=> (book-hotel ?dst ?start ?end ?money-left) (a hotel ?hotel)) (is ?money-left (- ?money ?flight-cost)))
				    (or (then (<= (hotel-booking ?hotel-cost) (a hotel ?hotel))
					      (<-- (=> (booking ?start ?end ?total-cost) (a ?booker ?b)) (is ?total-cost (+ ?flight-cost ?hotel-cost))))
					(then (<= no-vacancy (a hotel ?hotel))
					      (=> failure (a ?booker ?b)))))))))))

(define-role hotel-then-flight
    (== (a travel-agent ?ta)
	(then (<= (book-holiday ?src ?dst ?start ?end ?money) (a ?booker ?b))
	      (then (=> (book-hotel ?dst ?start ?end ?money) (a hotel ?hotel))
		    (or (then (<= no-vacancy (a hotel ?hotel))
			      (=> failure (a ?booker ?b)))
			(then (<= (hotel-booking ?hotel-cost) (a hotel ?hotel))
			      (then (<-- (=> (book-flight ?src ?dst ?start ?end ?money-left) (a airline ?airline)) (is ?money-left (- ?money ?hotel-cost)))
				    (or (then (<= no-flights (a airline ?airline))
					      (=> failure (a ?booker ?b)))
					(then (<= (flight-booking ?flight-cost) (a airline ?airline))
					      (<-- (=> (booking ?start ?end ?total-cost) (a ?booker ?b)) (is ?total-cost (+ ?flight-cost ?hotel-cost))))))))))))


(define-role matchmaker
    (== (a matchmaker matchmaker)
	(<-- (record-matchmaking-outcome! ?outcome)
	     (<= (matchmaking ?outcome) (a ? ?)))))

;;; }}}

;;; {{{ Agent definitions

(<-- (agents-for-role holiday ?role ?agents)
     (setof ?agent (agent-role ?agent ?role) ?agents))

(<-- (framework-agents holiday ?agents)
     (setof (a ?role ?agent) (agent-role ?agent ?role) ?service-agents)
     (append ?service-agents
	     ((a matchmaker matchmaker))
	     ?agents))

(<-- (agent-role ?agent ?role)
     (agent ?agent ?roles)
     (member ?role ?roles))

(<-- (agent dave (traveller)))
(<- (agent nicola (travel-agent)))
(<- (agent klm (airline)))
(<- (agent hilton (hotel)))

(<-- (travel-details edinburgh moldova 20 25 ?money)
     (threshold ?money))



;; Flights are relatively fixed (absolutely, here!) price.  No room
;; for manoeuvre.
(<-- (flight-available ?src ?dst ?start ?end ?money ?cost)
     (= ?cost 300)
     (not (> ?cost ?money))
     !)

(<-- (room-charge 200))

;; Hotels are more flexible...  
(<-- (room-available ?dst ?start ?end ?money ?cost)
     (threshold ?t)
     (= ?money ?t)
     ;; Eat a large chunk of the money, to make the flight difficult
     (random 0.4 0.7 ?r)
     (room-charge ?room-charge)
     (is ?cost (* ?r ?room-charge))
     (not (> ?cost ?money))
     !)
(<- (room-available ?dst ?start ?end ?money ?cost)
     (threshold ?t)
     (not (= ?money ?t))
     ;; Eat a small chunk of the money, since the flight is already booked
     (random 0.2 0.6 ?r)
     (room-charge ?room-charge)
     (is ?cost (* ?r ?room-charge))
     (not (> ?cost ?money))
     !)


(in-package :paiprolog)
(define-pill-function compute-flight-probability/2 (tolerance probability cont)
  (let ((tol (deref-exp tolerance)))
    (unify! probability (if (< tol 0)
			    (the-parabolic-f tol)
			    1)))
  (funcall cont))

(define-pill-function compute-hotel-probability/2 (tolerance probability cont)
  (let ((tol (deref-exp tolerance)))
    (unify! probability (if (> tol 0)
			    (the-parabolic-f tol)
			    1)))
  (funcall cont))

(defun the-parabolic-f (x)
  (- 1 (/ (expt (- x) 2) 5)))

;; One looks like: plot [0:7] 1 - ((7 - x)**2) / 49
;; and the other like: plot [0:7] ((7 - x)**2) / 49

#| 

  (== (a hotel ?h)
      (then (<= book-hotel ?dst ?start ?end ?tolerance) (a ?r ?a)
	    (or (<-- (=> (hotel-booking ?reservation-number) (a ?r ?a))
		     (room-available ?dst ?start ?end))
		(=> no-vacancy (a ?r ?a)))))

  (== (a airline ?a)
      (then (<= (book-flight ?src ?dst ?start ?end) (a ?r ?a))
	    (or (<-- (=> (flight-booking ?reservation-number) (a ?r ?a))
		     (flight-available ?src ?dst ?start ?end))
		(=> no-flights (a ?r ?a)))))


(<- (grid-ftp-send ?filename ?src ?dst bad))

(<-- (visualising ?))

(<-- (black-hole-present good heavy) 
     (random 0.0 1.0 ?t) 
     (msg black-hole-found/2 "?t=~W~%" (?t)) 
     (< ?t 0.8)
     !)

(<-- (serve-visualisation ??thing ??client))
(<-- (receive-visualisation ??thing ??server))
(<-- (file cygnus-x1))


(<-- (agents-for-role astrogrid ?role ?agents)
     (setof ?agent (agent-role ?agent ?role) ?agents))

|#
;;; }}}
